"use client"

import { useState, useEffect } from "react"
import Link from "next/link"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { ArrowLeft, MapPin, Phone, CheckCircle, Navigation } from "lucide-react"

interface Assignment {
  id: string
  requestId: string
  name: string
  phone: string
  vehicleType: string
  description: string
  location: { lat: number; lng: number }
  status: "assigned" | "in_progress" | "completed"
  assignedAt: string
}

export function TeamDashboard() {
  const [assignments, setAssignments] = useState<Assignment[]>([])

  useEffect(() => {
    // Mock assignments for rescue team
    const mockAssignments: Assignment[] = [
      {
        id: "asg001",
        requestId: "req002",
        name: "Trần Thị B",
        phone: "0987654321",
        vehicleType: "Xe máy",
        description: "Tai nạn nhỏ, cần hỗ trợ",
        location: { lat: 21.0245, lng: 105.8412 },
        status: "assigned",
        assignedAt: new Date().toISOString(),
      },
      {
        id: "asg002",
        requestId: "req003",
        name: "Lê Văn C",
        phone: "0901234567",
        vehicleType: "Xe tải",
        description: "Cần kéo xe về garage",
        location: { lat: 21.0335, lng: 105.85 },
        status: "in_progress",
        assignedAt: new Date(Date.now() - 1800000).toISOString(),
      },
    ]
    setAssignments(mockAssignments)
  }, [])

  const updateAssignmentStatus = (id: string, newStatus: Assignment["status"]) => {
    setAssignments((prev) => prev.map((asg) => (asg.id === id ? { ...asg, status: newStatus } : asg)))
  }

  const activeAssignments = assignments.filter((a) => a.status !== "completed")
  const completedAssignments = assignments.filter((a) => a.status === "completed")

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <header className="bg-white border-b sticky top-0 z-50">
        <div className="container mx-auto px-4 py-4">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold">Bảng điều khiển đội cứu hộ</h1>
              <p className="text-sm text-muted-foreground">Đội 1 - Hà Nội</p>
            </div>
            <Link href="/">
              <Button variant="outline">
                <ArrowLeft className="w-4 h-4 mr-2" />
                Trang chủ
              </Button>
            </Link>
          </div>
        </div>
      </header>

      <div className="container mx-auto px-4 py-8">
        {/* Stats */}
        <div className="grid sm:grid-cols-3 gap-4 mb-8">
          <Card>
            <CardHeader className="pb-2">
              <CardDescription>Nhiệm vụ đang chờ</CardDescription>
              <CardTitle className="text-3xl text-blue-600">
                {assignments.filter((a) => a.status === "assigned").length}
              </CardTitle>
            </CardHeader>
          </Card>
          <Card>
            <CardHeader className="pb-2">
              <CardDescription>Đang thực hiện</CardDescription>
              <CardTitle className="text-3xl text-orange-600">
                {assignments.filter((a) => a.status === "in_progress").length}
              </CardTitle>
            </CardHeader>
          </Card>
          <Card>
            <CardHeader className="pb-2">
              <CardDescription>Hoàn thành hôm nay</CardDescription>
              <CardTitle className="text-3xl text-green-600">{completedAssignments.length}</CardTitle>
            </CardHeader>
          </Card>
        </div>

        {/* Active Assignments */}
        <div className="mb-8">
          <h2 className="text-xl font-semibold mb-4">Nhiệm vụ hiện tại</h2>
          <div className="space-y-4">
            {activeAssignments.map((assignment) => (
              <Card key={assignment.id} className="border-l-4 border-l-red-600">
                <CardHeader>
                  <div className="flex items-start justify-between">
                    <div>
                      <CardTitle className="text-lg">{assignment.name}</CardTitle>
                      <CardDescription>
                        Mã: {assignment.requestId} • {new Date(assignment.assignedAt).toLocaleString("vi-VN")}
                      </CardDescription>
                    </div>
                    <Badge
                      variant="secondary"
                      className={
                        assignment.status === "assigned" ? "bg-blue-500 text-white" : "bg-orange-500 text-white"
                      }
                    >
                      {assignment.status === "assigned" ? "Chờ xuất phát" : "Đang cứu hộ"}
                    </Badge>
                  </div>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div className="grid sm:grid-cols-2 gap-3 text-sm">
                    <div className="flex items-center gap-2">
                      <Phone className="w-4 h-4 text-muted-foreground" />
                      <span>{assignment.phone}</span>
                    </div>
                    <div className="flex items-center gap-2">
                      <span className="text-muted-foreground">Phương tiện:</span>
                      <span className="font-medium">{assignment.vehicleType}</span>
                    </div>
                    <div className="sm:col-span-2 flex items-start gap-2">
                      <MapPin className="w-4 h-4 text-muted-foreground mt-0.5" />
                      <div>
                        <p className="font-medium">
                          {assignment.location.lat.toFixed(6)}, {assignment.location.lng.toFixed(6)}
                        </p>
                        <Button variant="link" className="h-auto p-0 text-blue-600" size="sm">
                          <Navigation className="w-3 h-3 mr-1" />
                          Mở Google Maps
                        </Button>
                      </div>
                    </div>
                  </div>
                  <div>
                    <p className="text-sm text-muted-foreground mb-1">Mô tả:</p>
                    <p className="text-sm">{assignment.description}</p>
                  </div>
                  <div className="flex gap-2 pt-2">
                    {assignment.status === "assigned" && (
                      <Button
                        className="bg-orange-600 hover:bg-orange-700"
                        onClick={() => updateAssignmentStatus(assignment.id, "in_progress")}
                      >
                        Bắt đầu cứu hộ
                      </Button>
                    )}
                    {assignment.status === "in_progress" && (
                      <Button
                        className="bg-green-600 hover:bg-green-700"
                        onClick={() => updateAssignmentStatus(assignment.id, "completed")}
                      >
                        <CheckCircle className="w-4 h-4 mr-2" />
                        Hoàn thành
                      </Button>
                    )}
                    <Button variant="outline">
                      <Phone className="w-4 h-4 mr-2" />
                      Gọi khách hàng
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}

            {activeAssignments.length === 0 && (
              <Card className="p-12">
                <div className="text-center text-muted-foreground">
                  <CheckCircle className="w-12 h-12 mx-auto mb-4 opacity-50" />
                  <p>Không có nhiệm vụ nào đang chờ xử lý</p>
                </div>
              </Card>
            )}
          </div>
        </div>

        {/* Completed Today */}
        {completedAssignments.length > 0 && (
          <div>
            <h2 className="text-xl font-semibold mb-4">Đã hoàn thành hôm nay</h2>
            <div className="space-y-2">
              {completedAssignments.map((assignment) => (
                <Card key={assignment.id}>
                  <CardContent className="py-4">
                    <div className="flex items-center justify-between">
                      <div>
                        <p className="font-medium">{assignment.name}</p>
                        <p className="text-sm text-muted-foreground">{assignment.requestId}</p>
                      </div>
                      <Badge variant="secondary" className="bg-green-500 text-white">
                        Hoàn thành
                      </Badge>
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        )}
      </div>
    </div>
  )
}
