"use client"

import type React from "react"

import { useState, useEffect } from "react"
import { useRouter } from "next/navigation"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { Textarea } from "@/components/ui/textarea"
import { MapPin, Phone, Upload, AlertCircle, Waves } from "lucide-react"
import { Alert, AlertDescription } from "@/components/ui/alert"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Checkbox } from "@/components/ui/checkbox"
import { getCurrentUser, isAuthenticated } from "@/lib/auth"

export function RescueRequestForm() {
  const router = useRouter()
  const [loading, setLoading] = useState(false)
  const [location, setLocation] = useState<{ lat: number; lng: number } | null>(null)
  const [formData, setFormData] = useState({
    name: "",
    phone: "",
    address: "",
    numberOfPeople: "1",
    waterLevel: "",
    description: "",
    urgentNeeds: "",
    hasChildren: false,
    hasElderly: false,
    hasPets: false,
  })

  useEffect(() => {
    if (!isAuthenticated()) {
      router.push("/login")
      return
    }
    const user = getCurrentUser()
    if (user) {
      setFormData((prev) => ({
        ...prev,
        name: user.fullName,
        phone: user.phone,
      }))
    }
  }, [router])

  const getLocation = () => {
    if ("geolocation" in navigator) {
      navigator.geolocation.getCurrentPosition(
        (position) => {
          setLocation({
            lat: position.coords.latitude,
            lng: position.coords.longitude,
          })
        },
        (error) => {
          console.error("Error getting location:", error)
          alert("Không thể lấy vị trí. Vui lòng bật GPS và thử lại.")
        },
      )
    }
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setLoading(true)

    const user = getCurrentUser()
    if (!user) {
      router.push("/login")
      return
    }

    // Simulate API call
    setTimeout(() => {
      const requestId = Math.random().toString(36).substring(7)
      localStorage.setItem("lastRequestId", requestId)

      // Save to mock database
      const requests = JSON.parse(localStorage.getItem("rescueRequests") || "[]")
      let priority = "medium"
      if (formData.waterLevel === "chest" || formData.waterLevel === "roof") {
        priority = "critical"
      } else if (formData.waterLevel === "waist" || formData.hasChildren || formData.hasElderly) {
        priority = "high"
      }

      requests.push({
        id: requestId,
        userId: user.id,
        ...formData,
        location,
        status: "pending",
        priority,
        createdAt: new Date().toISOString(),
      })
      localStorage.setItem("rescueRequests", JSON.stringify(requests))

      setLoading(false)
      router.push(`/request/success?id=${requestId}`)
    }, 1000)
  }

  return (
    <div className="max-w-2xl mx-auto">
      <Card>
        <CardHeader>
          <CardTitle className="text-2xl text-blue-600 flex items-center gap-2">
            <Waves className="w-6 h-6" />
            Yêu cầu cứu hộ lũ lụt khẩn cấp
          </CardTitle>
          <CardDescription>Điền thông tin chi tiết để chúng tôi có thể hỗ trợ bạn nhanh nhất</CardDescription>
        </CardHeader>
        <CardContent>
          <form onSubmit={handleSubmit} className="space-y-6">
            {/* Location */}
            <div className="space-y-2">
              <Label>Vị trí của bạn</Label>
              <Button
                type="button"
                variant="outline"
                className="w-full justify-start bg-transparent"
                onClick={getLocation}
              >
                <MapPin className="w-4 h-4 mr-2" />
                {location
                  ? `Đã lấy vị trí: ${location.lat.toFixed(6)}, ${location.lng.toFixed(6)}`
                  : "Lấy vị trí hiện tại"}
              </Button>
              {!location && (
                <Alert>
                  <AlertCircle className="h-4 w-4" />
                  <AlertDescription>Vui lòng cho phép truy cập vị trí để chúng tôi tìm được bạn</AlertDescription>
                </Alert>
              )}
            </div>

            {/* Contact Info */}
            <div className="grid sm:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="name">Họ và tên</Label>
                <Input
                  id="name"
                  placeholder="Nguyễn Văn A"
                  value={formData.name}
                  onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                  required
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="phone">Số điện thoại</Label>
                <Input
                  id="phone"
                  type="tel"
                  placeholder="0912345678"
                  value={formData.phone}
                  onChange={(e) => setFormData({ ...formData, phone: e.target.value })}
                  required
                />
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="address">Địa chỉ cụ thể</Label>
              <Input
                id="address"
                placeholder="Số nhà, tên đường, phường/xã..."
                value={formData.address}
                onChange={(e) => setFormData({ ...formData, address: e.target.value })}
                required
              />
            </div>

            <div className="grid sm:grid-cols-2 gap-4">
              <div className="space-y-2">
                <Label htmlFor="numberOfPeople">Số người cần cứu</Label>
                <Select
                  value={formData.numberOfPeople}
                  onValueChange={(value) => setFormData({ ...formData, numberOfPeople: value })}
                >
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="1">1 người</SelectItem>
                    <SelectItem value="2">2 người</SelectItem>
                    <SelectItem value="3">3 người</SelectItem>
                    <SelectItem value="4">4 người</SelectItem>
                    <SelectItem value="5+">5+ người</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div className="space-y-2">
                <Label htmlFor="waterLevel">Mức nước hiện tại</Label>
                <Select
                  value={formData.waterLevel}
                  onValueChange={(value) => setFormData({ ...formData, waterLevel: value })}
                  required
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Chọn mức nước" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="ankle">Ngập đến mắt cá</SelectItem>
                    <SelectItem value="knee">Ngập đến đầu gối</SelectItem>
                    <SelectItem value="waist">Ngập đến thắt lưng</SelectItem>
                    <SelectItem value="chest">Ngập đến ngực</SelectItem>
                    <SelectItem value="roof">Phải trèo lên mái nhà</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>

            <div className="space-y-3">
              <Label>Người cần ưu tiên</Label>
              <div className="space-y-2">
                <div className="flex items-center space-x-2">
                  <Checkbox
                    id="hasChildren"
                    checked={formData.hasChildren}
                    onCheckedChange={(checked) => setFormData({ ...formData, hasChildren: checked as boolean })}
                  />
                  <label htmlFor="hasChildren" className="text-sm cursor-pointer">
                    Có trẻ em
                  </label>
                </div>
                <div className="flex items-center space-x-2">
                  <Checkbox
                    id="hasElderly"
                    checked={formData.hasElderly}
                    onCheckedChange={(checked) => setFormData({ ...formData, hasElderly: checked as boolean })}
                  />
                  <label htmlFor="hasElderly" className="text-sm cursor-pointer">
                    Có người cao tuổi
                  </label>
                </div>
                <div className="flex items-center space-x-2">
                  <Checkbox
                    id="hasPets"
                    checked={formData.hasPets}
                    onCheckedChange={(checked) => setFormData({ ...formData, hasPets: checked as boolean })}
                  />
                  <label htmlFor="hasPets" className="text-sm cursor-pointer">
                    Có thú cưng
                  </label>
                </div>
              </div>
            </div>

            {/* Description */}
            <div className="space-y-2">
              <Label htmlFor="description">Mô tả tình huống</Label>
              <Textarea
                id="description"
                placeholder="Mô tả chi tiết tình trạng ngập nước, có bị mắc kẹt không, có thể di chuyển không..."
                rows={4}
                value={formData.description}
                onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                required
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="urgentNeeds">Nhu cầu cấp thiết (nếu có)</Label>
              <Textarea
                id="urgentNeeds"
                placeholder="Thực phẩm, nước uống, thuốc men..."
                rows={2}
                value={formData.urgentNeeds}
                onChange={(e) => setFormData({ ...formData, urgentNeeds: e.target.value })}
              />
            </div>

            {/* Image Upload */}
            <div className="space-y-2">
              <Label htmlFor="images">Hình ảnh (nếu có)</Label>
              <div className="border-2 border-dashed rounded-lg p-8 text-center hover:border-blue-300 transition-colors">
                <Upload className="w-8 h-8 mx-auto mb-2 text-muted-foreground" />
                <p className="text-sm text-muted-foreground">Nhấn để tải lên hoặc kéo thả hình ảnh</p>
                <Input id="images" type="file" className="hidden" multiple accept="image/*" />
              </div>
            </div>

            {/* Submit */}
            <Button
              type="submit"
              className="w-full bg-blue-600 hover:bg-blue-700"
              size="lg"
              disabled={loading || !location || !formData.waterLevel}
            >
              <Phone className="w-4 h-4 mr-2" />
              {loading ? "Đang gửi yêu cầu..." : "Gửi yêu cầu cứu hộ"}
            </Button>
          </form>
        </CardContent>
      </Card>
    </div>
  )
}
