"use client"

import { useEffect, useState } from "react"
import { Card } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Navigation, Waves, AlertCircle } from "lucide-react"

interface RescueRequest {
  id: string
  location: { lat: number; lng: number }
  address: string
  status: "pending" | "assigned" | "in_progress" | "completed"
  name: string
  waterLevel: string
  peopleCount: number
  hasChildren: boolean
  hasElderly: boolean
}

export function RescueMap() {
  const [requests, setRequests] = useState<RescueRequest[]>([])
  const [selectedRequest, setSelectedRequest] = useState<RescueRequest | null>(null)

  useEffect(() => {
    const stored = localStorage.getItem("rescueRequests")
    if (stored) {
      const allRequests = JSON.parse(stored)
      setRequests(allRequests)
    } else {
      // Mock data nếu chưa có dữ liệu
      const mockRequests: RescueRequest[] = [
        {
          id: "1",
          location: { lat: 21.0285, lng: 105.8542 },
          address: "Số 10 Đường Lê Duẩn, Quận Hoàn Kiếm, Hà Nội",
          status: "pending",
          name: "Nguyễn Văn A",
          waterLevel: "over_1m",
          peopleCount: 4,
          hasChildren: true,
          hasElderly: false,
        },
        {
          id: "2",
          location: { lat: 21.0245, lng: 105.8412 },
          address: "Số 25 Phố Huế, Quận Hai Bà Trưng, Hà Nội",
          status: "assigned",
          name: "Trần Thị B",
          waterLevel: "50cm_1m",
          peopleCount: 2,
          hasChildren: false,
          hasElderly: true,
        },
        {
          id: "3",
          location: { lat: 21.0335, lng: 105.85 },
          address: "Số 5 Đường Giải Phóng, Quận Đống Đa, Hà Nội",
          status: "in_progress",
          name: "Lê Văn C",
          waterLevel: "30_50cm",
          peopleCount: 6,
          hasChildren: true,
          hasElderly: true,
        },
      ]
      setRequests(mockRequests)
    }

    // Auto-refresh every 10 seconds
    const interval = setInterval(() => {
      const stored = localStorage.getItem("rescueRequests")
      if (stored) {
        setRequests(JSON.parse(stored))
      }
    }, 10000)

    return () => clearInterval(interval)
  }, [])

  const getStatusColor = (status: string) => {
    switch (status) {
      case "pending":
        return "bg-red-500"
      case "assigned":
        return "bg-blue-500"
      case "in_progress":
        return "bg-orange-500"
      case "completed":
        return "bg-green-500"
      default:
        return "bg-gray-500"
    }
  }

  const getStatusText = (status: string) => {
    switch (status) {
      case "pending":
        return "Khẩn cấp"
      case "assigned":
        return "Đã phân công"
      case "in_progress":
        return "Đang cứu hộ"
      case "completed":
        return "Hoàn thành"
      default:
        return status
    }
  }

  const getWaterLevelText = (level: string) => {
    switch (level) {
      case "below_30cm":
        return "Dưới 30cm"
      case "30_50cm":
        return "30-50cm"
      case "50cm_1m":
        return "50cm-1m"
      case "over_1m":
        return "Trên 1m"
      default:
        return level
    }
  }

  return (
    <div className="grid lg:grid-cols-3 gap-4">
      {/* Map Area */}
      <div className="lg:col-span-2 space-y-4">
        <Card className="h-[600px] bg-gradient-to-br from-blue-50 to-cyan-50 relative overflow-hidden border-2 border-blue-200">
          <div
            className="absolute inset-0 opacity-10"
            style={{
              backgroundImage: `repeating-linear-gradient(0deg, #0ea5e9 0px, #0ea5e9 1px, transparent 1px, transparent 40px),
                                   repeating-linear-gradient(90deg, #0ea5e9 0px, #0ea5e9 1px, transparent 1px, transparent 40px)`,
            }}
          />

          <div className="absolute top-4 left-4 right-4 bg-white/90 backdrop-blur-sm rounded-lg p-3 shadow-lg">
            <div className="flex items-center gap-2 text-sm">
              <AlertCircle className="w-4 h-4 text-blue-600" />
              <span className="font-medium">Bản đồ cứu hộ lũ lụt - {requests.length} yêu cầu đang theo dõi</span>
            </div>
          </div>

          {/* Map markers */}
          {requests.map((req, idx) => (
            <button
              key={req.id}
              onClick={() => setSelectedRequest(req)}
              className={`absolute transform -translate-x-1/2 -translate-y-full cursor-pointer hover:scale-110 transition-transform ${
                selectedRequest?.id === req.id ? "z-20 scale-125" : "z-10"
              }`}
              style={{
                left: `${25 + idx * 22}%`,
                top: `${35 + (idx % 2) * 25}%`,
              }}
            >
              <div className="relative">
                <div
                  className={`w-8 h-8 ${getStatusColor(req.status)} rounded-full border-3 border-white shadow-2xl flex items-center justify-center animate-pulse`}
                >
                  <Waves className="w-4 h-4 text-white" />
                </div>
                {req.hasChildren || req.hasElderly ? (
                  <div className="absolute -top-1 -right-1 w-3 h-3 bg-yellow-500 rounded-full border border-white" />
                ) : null}
              </div>
            </button>
          ))}

          {/* Selected request popup */}
          {selectedRequest && (
            <div className="absolute bottom-8 left-1/2 transform -translate-x-1/2 z-30 w-80 max-w-[calc(100%-2rem)]">
              <Card className="p-4 shadow-2xl border-2 border-blue-300 bg-white">
                <div className="space-y-3">
                  <div className="flex items-start justify-between">
                    <div>
                      <p className="font-bold text-lg">{selectedRequest.name}</p>
                      <p className="text-sm text-muted-foreground">{selectedRequest.address}</p>
                    </div>
                    <Badge className={getStatusColor(selectedRequest.status) + " text-white"}>
                      {getStatusText(selectedRequest.status)}
                    </Badge>
                  </div>

                  <div className="grid grid-cols-2 gap-2 text-sm">
                    <div className="flex items-center gap-1">
                      <Waves className="w-4 h-4 text-blue-600" />
                      <span>{getWaterLevelText(selectedRequest.waterLevel)}</span>
                    </div>
                    <div className="flex items-center gap-1">
                      <span className="font-medium">{selectedRequest.peopleCount} người</span>
                    </div>
                  </div>

                  {(selectedRequest.hasChildren || selectedRequest.hasElderly) && (
                    <div className="flex gap-2">
                      {selectedRequest.hasChildren && (
                        <Badge variant="outline" className="text-xs bg-yellow-50">
                          Có trẻ em
                        </Badge>
                      )}
                      {selectedRequest.hasElderly && (
                        <Badge variant="outline" className="text-xs bg-orange-50">
                          Có người già
                        </Badge>
                      )}
                    </div>
                  )}

                  <Button
                    size="sm"
                    className="w-full bg-blue-600 hover:bg-blue-700"
                    onClick={() =>
                      window.open(
                        `https://www.google.com/maps?q=${selectedRequest.location.lat},${selectedRequest.location.lng}`,
                        "_blank",
                      )
                    }
                  >
                    <Navigation className="w-4 h-4 mr-2" />
                    Chỉ đường
                  </Button>
                </div>
              </Card>
            </div>
          )}
        </Card>

        {/* Legend */}
        <Card className="p-4">
          <p className="font-semibold mb-3">Chú thích</p>
          <div className="grid grid-cols-2 sm:grid-cols-4 gap-3">
            <div className="flex items-center gap-2">
              <div className="w-4 h-4 bg-red-500 rounded-full" />
              <span className="text-sm">Khẩn cấp</span>
            </div>
            <div className="flex items-center gap-2">
              <div className="w-4 h-4 bg-blue-500 rounded-full" />
              <span className="text-sm">Đã phân công</span>
            </div>
            <div className="flex items-center gap-2">
              <div className="w-4 h-4 bg-orange-500 rounded-full" />
              <span className="text-sm">Đang cứu hộ</span>
            </div>
            <div className="flex items-center gap-2">
              <div className="w-4 h-4 bg-green-500 rounded-full" />
              <span className="text-sm">Hoàn thành</span>
            </div>
          </div>
        </Card>
      </div>

      {/* Request List Sidebar */}
      <div className="space-y-3">
        <Card className="p-4 bg-blue-50">
          <h3 className="font-semibold mb-1">Danh sách yêu cầu</h3>
          <p className="text-sm text-muted-foreground">Nhấn vào marker hoặc yêu cầu để xem chi tiết</p>
        </Card>

        <div className="space-y-2 max-h-[650px] overflow-y-auto">
          {requests.map((req) => (
            <Card
              key={req.id}
              className={`p-3 cursor-pointer transition-all hover:shadow-md ${
                selectedRequest?.id === req.id ? "ring-2 ring-blue-500 bg-blue-50" : ""
              }`}
              onClick={() => setSelectedRequest(req)}
            >
              <div className="space-y-2">
                <div className="flex items-start justify-between gap-2">
                  <div className="flex items-start gap-2 flex-1 min-w-0">
                    <Waves className="w-5 h-5 text-blue-600 flex-shrink-0 mt-0.5" />
                    <div className="min-w-0 flex-1">
                      <p className="font-medium truncate">{req.name}</p>
                      <p className="text-xs text-muted-foreground line-clamp-2">{req.address}</p>
                    </div>
                  </div>
                  <Badge
                    variant="secondary"
                    className={getStatusColor(req.status) + " text-white text-xs flex-shrink-0"}
                  >
                    {getStatusText(req.status)}
                  </Badge>
                </div>

                <div className="flex items-center gap-2 text-xs text-muted-foreground">
                  <span>{getWaterLevelText(req.waterLevel)}</span>
                  <span>•</span>
                  <span>{req.peopleCount} người</span>
                </div>

                {(req.hasChildren || req.hasElderly) && (
                  <div className="flex gap-1">
                    {req.hasChildren && (
                      <Badge variant="outline" className="text-xs bg-yellow-50 px-1 py-0">
                        Trẻ em
                      </Badge>
                    )}
                    {req.hasElderly && (
                      <Badge variant="outline" className="text-xs bg-orange-50 px-1 py-0">
                        Người già
                      </Badge>
                    )}
                  </div>
                )}
              </div>
            </Card>
          ))}
        </div>
      </div>
    </div>
  )
}
