# Hướng dẫn cấu hình cơ sở dữ liệu MySQL

## Thông tin kết nối

Thêm các biến môi trường sau vào dự án Vercel hoặc file `.env.local`:

\`\`\`
DATABASE_URL=mysql://username:password@host:3306/database_name
DB_HOST=localhost
DB_USER=root
DB_PASSWORD=your_password
DB_NAME=flood_rescue
\`\`\`

## Cài đặt database

### 1. Tạo database

\`\`\`sql
CREATE DATABASE flood_rescue CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE flood_rescue;
\`\`\`

### 2. Chạy schema

Chạy file `scripts/database-schema.sql` trong MySQL:

\`\`\`bash
mysql -u root -p flood_rescue < scripts/database-schema.sql
\`\`\`

Hoặc copy nội dung file vào MySQL Workbench/phpMyAdmin và thực thi.

## Schema bảng

### Bảng `users`
Lưu thông tin người dùng và phân quyền:
- `id`: UUID primary key
- `email`: Email đăng nhập (unique)
- `password_hash`: Mật khẩu đã mã hóa bcrypt
- `full_name`: Họ tên đầy đủ
- `phone`: Số điện thoại
- `role`: Vai trò (victim/admin)
- `created_at`: Thời gian tạo

### Bảng `rescue_requests`
Lưu các yêu cầu cứu hộ:
- `id`: UUID primary key
- `user_id`: ID người yêu cầu (có thể NULL nếu chưa đăng nhập)
- `full_name`: Tên người cần cứu hộ
- `phone`: SĐT liên lạc
- `address`: Địa chỉ chi tiết
- `latitude/longitude`: Tọa độ GPS
- `water_level`: Mức nước (ankle/knee/waist/chest/overhead)
- `people_count`: Số người cần cứu hộ
- `has_children`: Có trẻ em không
- `has_elderly`: Có người già không
- `description`: Mô tả tình huống
- `images`: JSON array chứa URL ảnh
- `status`: Trạng thái (pending/assigned/inprogress/completed/cancelled)
- `priority`: Độ ưu tiên (low/medium/high/urgent)
- `assigned_to`: ID đội cứu hộ được phân công
- `created_at/updated_at`: Timestamps

## Kết nối từ Node.js

### Cài đặt package

\`\`\`bash
npm install mysql2
\`\`\`

### Code mẫu kết nối

\`\`\`javascript
// lib/db.ts
import mysql from 'mysql2/promise'

const pool = mysql.createPool({
  host: process.env.DB_HOST,
  user: process.env.DB_USER,
  password: process.env.DB_PASSWORD,
  database: process.env.DB_NAME,
  waitForConnections: true,
  connectionLimit: 10,
  queueLimit: 0
})

export default pool
\`\`\`

### API Route mẫu

\`\`\`typescript
// app/api/rescue-requests/route.ts
import pool from '@/lib/db'
import { NextResponse } from 'next/server'

export async function POST(request: Request) {
  const data = await request.json()
  
  const [result] = await pool.execute(
    `INSERT INTO rescue_requests 
    (id, full_name, phone, address, latitude, longitude, water_level, people_count, description, status) 
    VALUES (UUID(), ?, ?, ?, ?, ?, ?, ?, ?, 'pending')`,
    [data.fullName, data.phone, data.address, data.latitude, data.longitude, data.waterLevel, data.peopleCount, data.description]
  )
  
  return NextResponse.json({ success: true, result })
}
\`\`\`

## Bảo mật

- Luôn sử dụng `bcrypt` để hash mật khẩu trước khi lưu
- Không bao giờ lưu password dạng plain text
- Sử dụng prepared statements để tránh SQL injection
- Giới hạn rate limiting cho các API endpoint
- Xác thực JWT token trước khi cho phép truy cập admin

## Migration (khi cần thay đổi schema)

Tạo file SQL mới với prefix timestamp:

\`\`\`
scripts/migration-001-add-team-field.sql
scripts/migration-002-add-status-index.sql
\`\`\`

Chạy theo thứ tự để đảm bảo database luôn đúng phiên bản.
