"use client"

import { useState, useEffect } from "react"
import Link from "next/link"
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card"
import { Button } from "@/components/ui/button"
import { Badge } from "@/components/ui/badge"
import { Input } from "@/components/ui/input"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { AlertCircle, CheckCircle, Clock, Users, ArrowLeft, Search } from "lucide-react"

interface RescueRequest {
  id: string
  name: string
  phone: string
  vehicleType: string
  description: string
  status: "pending" | "assigned" | "in_progress" | "completed"
  createdAt: string
  location: { lat: number; lng: number } | null
  assignedTeam?: string
}

export function AdminDashboard() {
  const [requests, setRequests] = useState<RescueRequest[]>([])
  const [searchTerm, setSearchTerm] = useState("")
  const [filterStatus, setFilterStatus] = useState<string>("all")

  useEffect(() => {
    // Load requests from localStorage
    const stored = localStorage.getItem("rescueRequests")
    if (stored) {
      setRequests(JSON.parse(stored))
    } else {
      // Mock data
      const mockData: RescueRequest[] = [
        {
          id: "req001",
          name: "Nguyễn Văn A",
          phone: "0912345678",
          vehicleType: "Ô tô",
          description: "Xe bị hỏng máy trên cao tốc",
          status: "pending",
          createdAt: new Date().toISOString(),
          location: { lat: 21.0285, lng: 105.8542 },
        },
        {
          id: "req002",
          name: "Trần Thị B",
          phone: "0987654321",
          vehicleType: "Xe máy",
          description: "Tai nạn nhỏ, cần hỗ trợ",
          status: "assigned",
          createdAt: new Date(Date.now() - 3600000).toISOString(),
          location: { lat: 21.0245, lng: 105.8412 },
          assignedTeam: "Đội 1 - Hà Nội",
        },
        {
          id: "req003",
          name: "Lê Văn C",
          phone: "0901234567",
          vehicleType: "Xe tải",
          description: "Cần kéo xe về garage",
          status: "in_progress",
          createdAt: new Date(Date.now() - 7200000).toISOString(),
          location: { lat: 21.0335, lng: 105.85 },
          assignedTeam: "Đội 2 - Hà Nội",
        },
      ]
      setRequests(mockData)
      localStorage.setItem("rescueRequests", JSON.stringify(mockData))
    }
  }, [])

  const updateRequestStatus = (id: string, newStatus: RescueRequest["status"], team?: string) => {
    const updated = requests.map((req) =>
      req.id === id ? { ...req, status: newStatus, assignedTeam: team || req.assignedTeam } : req,
    )
    setRequests(updated)
    localStorage.setItem("rescueRequests", JSON.stringify(updated))
  }

  const getStatusBadge = (status: string) => {
    switch (status) {
      case "pending":
        return (
          <Badge variant="secondary" className="bg-yellow-500 text-white">
            Chờ xử lý
          </Badge>
        )
      case "assigned":
        return (
          <Badge variant="secondary" className="bg-blue-500 text-white">
            Đã phân công
          </Badge>
        )
      case "in_progress":
        return (
          <Badge variant="secondary" className="bg-orange-500 text-white">
            Đang cứu hộ
          </Badge>
        )
      case "completed":
        return (
          <Badge variant="secondary" className="bg-green-500 text-white">
            Hoàn thành
          </Badge>
        )
      default:
        return <Badge>{status}</Badge>
    }
  }

  const filteredRequests = requests.filter((req) => {
    const matchesSearch =
      req.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
      req.phone.includes(searchTerm) ||
      req.id.toLowerCase().includes(searchTerm.toLowerCase())
    const matchesFilter = filterStatus === "all" || req.status === filterStatus
    return matchesSearch && matchesFilter
  })

  const stats = {
    pending: requests.filter((r) => r.status === "pending").length,
    assigned: requests.filter((r) => r.status === "assigned").length,
    inProgress: requests.filter((r) => r.status === "in_progress").length,
    completed: requests.filter((r) => r.status === "completed").length,
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <header className="bg-white border-b sticky top-0 z-50">
        <div className="container mx-auto px-4 py-4">
          <div className="flex items-center justify-between">
            <div>
              <h1 className="text-2xl font-bold">Quản trị hệ thống cứu hộ</h1>
              <p className="text-sm text-muted-foreground">Quản lý yêu cầu và đội cứu hộ</p>
            </div>
            <Link href="/">
              <Button variant="outline">
                <ArrowLeft className="w-4 h-4 mr-2" />
                Trang chủ
              </Button>
            </Link>
          </div>
        </div>
      </header>

      <div className="container mx-auto px-4 py-8">
        {/* Stats */}
        <div className="grid sm:grid-cols-2 lg:grid-cols-4 gap-4 mb-8">
          <Card>
            <CardHeader className="pb-2">
              <CardDescription>Chờ xử lý</CardDescription>
              <CardTitle className="text-3xl text-yellow-600">{stats.pending}</CardTitle>
            </CardHeader>
            <CardContent>
              <Clock className="w-4 h-4 text-muted-foreground" />
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="pb-2">
              <CardDescription>Đã phân công</CardDescription>
              <CardTitle className="text-3xl text-blue-600">{stats.assigned}</CardTitle>
            </CardHeader>
            <CardContent>
              <AlertCircle className="w-4 h-4 text-muted-foreground" />
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="pb-2">
              <CardDescription>Đang cứu hộ</CardDescription>
              <CardTitle className="text-3xl text-orange-600">{stats.inProgress}</CardTitle>
            </CardHeader>
            <CardContent>
              <Users className="w-4 h-4 text-muted-foreground" />
            </CardContent>
          </Card>
          <Card>
            <CardHeader className="pb-2">
              <CardDescription>Hoàn thành</CardDescription>
              <CardTitle className="text-3xl text-green-600">{stats.completed}</CardTitle>
            </CardHeader>
            <CardContent>
              <CheckCircle className="w-4 h-4 text-muted-foreground" />
            </CardContent>
          </Card>
        </div>

        {/* Filters */}
        <Card className="mb-6">
          <CardContent className="pt-6">
            <div className="flex flex-col sm:flex-row gap-4">
              <div className="relative flex-1">
                <Search className="absolute left-3 top-3 w-4 h-4 text-muted-foreground" />
                <Input
                  placeholder="Tìm theo tên, SĐT, mã yêu cầu..."
                  className="pl-9"
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                />
              </div>
              <Select value={filterStatus} onValueChange={setFilterStatus}>
                <SelectTrigger className="w-full sm:w-[200px]">
                  <SelectValue placeholder="Lọc trạng thái" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Tất cả</SelectItem>
                  <SelectItem value="pending">Chờ xử lý</SelectItem>
                  <SelectItem value="assigned">Đã phân công</SelectItem>
                  <SelectItem value="in_progress">Đang cứu hộ</SelectItem>
                  <SelectItem value="completed">Hoàn thành</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </CardContent>
        </Card>

        {/* Requests List */}
        <div className="space-y-4">
          {filteredRequests.map((request) => (
            <Card key={request.id}>
              <CardHeader>
                <div className="flex items-start justify-between">
                  <div>
                    <CardTitle className="text-lg">{request.name}</CardTitle>
                    <CardDescription>
                      Mã: {request.id} • {new Date(request.createdAt).toLocaleString("vi-VN")}
                    </CardDescription>
                  </div>
                  {getStatusBadge(request.status)}
                </div>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid sm:grid-cols-2 gap-4 text-sm">
                  <div>
                    <span className="text-muted-foreground">Điện thoại:</span>
                    <span className="ml-2 font-medium">{request.phone}</span>
                  </div>
                  <div>
                    <span className="text-muted-foreground">Phương tiện:</span>
                    <span className="ml-2 font-medium">{request.vehicleType}</span>
                  </div>
                  {request.location && (
                    <div className="sm:col-span-2">
                      <span className="text-muted-foreground">Vị trí:</span>
                      <span className="ml-2 font-medium">
                        {request.location.lat.toFixed(6)}, {request.location.lng.toFixed(6)}
                      </span>
                    </div>
                  )}
                  {request.assignedTeam && (
                    <div className="sm:col-span-2">
                      <span className="text-muted-foreground">Đội cứu hộ:</span>
                      <span className="ml-2 font-medium">{request.assignedTeam}</span>
                    </div>
                  )}
                </div>
                <div>
                  <p className="text-sm text-muted-foreground mb-1">Mô tả:</p>
                  <p className="text-sm">{request.description}</p>
                </div>
                <div className="flex gap-2 pt-2">
                  {request.status === "pending" && (
                    <Button size="sm" onClick={() => updateRequestStatus(request.id, "assigned", "Đội 1 - Hà Nội")}>
                      Phân công đội
                    </Button>
                  )}
                  {request.status === "assigned" && (
                    <Button size="sm" onClick={() => updateRequestStatus(request.id, "in_progress")}>
                      Bắt đầu cứu hộ
                    </Button>
                  )}
                  {request.status === "in_progress" && (
                    <Button
                      size="sm"
                      className="bg-green-600 hover:bg-green-700"
                      onClick={() => updateRequestStatus(request.id, "completed")}
                    >
                      Hoàn thành
                    </Button>
                  )}
                  <Button size="sm" variant="outline">
                    Xem chi tiết
                  </Button>
                </div>
              </CardContent>
            </Card>
          ))}

          {filteredRequests.length === 0 && (
            <Card className="p-12">
              <div className="text-center text-muted-foreground">
                <AlertCircle className="w-12 h-12 mx-auto mb-4 opacity-50" />
                <p>Không tìm thấy yêu cầu nào</p>
              </div>
            </Card>
          )}
        </div>
      </div>
    </div>
  )
}
